# Load necessary libraries
library(survival)
library(openxlsx)

# Load the data
  data <- read.xlsx("filled_data age.xlsx")

# Clean column names by removing "X." prefix (if present)
colnames(data) <- gsub("^X\\.", "", colnames(data))  # This will remove "X." from the column names

# Convert the survival status to a binary numeric variable (if not already in numeric format)
data$AE <- as.numeric(data$AE)

# Define the survival object
surv_obj <- Surv(data$随访时间, data$AE)

# Initialize a data frame to store all results
all_results <- data.frame(Variable = character(), Beta = numeric(), HR = numeric(), CI_lower = numeric(), CI_upper = numeric(), p_value = numeric())

# Perform univariate Cox regression for each variable
for (var in names(data)[4:(ncol(data) - 1)]) {  # Exclude the survival status column
  # Ensure the variable is enclosed in backticks to handle special characters
  formula <- as.formula(paste("surv_obj ~ `", var, "`", sep=""))
  
  # Perform Cox regression
  cox_model <- try(coxph(formula, data = data), silent = TRUE)
  
  if (inherits(cox_model, "try-error")) {
    next  # Skip if the model fails to converge or produces an error
  }
  
  result <- summary(cox_model)
  
  # Extract β, HR, CI, and p-value
  beta <- result$coefficients[1, 1]  # Coefficient (β)
  hr <- exp(beta)  # Hazard ratio (HR)
  
  # Correct extraction of CI (Confidence Interval)
  ci_lower <- result$conf.int[1, 3]  # Lower bound of Confidence Interval
  ci_upper <- result$conf.int[1, 4]  # Upper bound of Confidence Interval
  p_value <- result$coefficients[1, 5]  # p-value
  
  # Append results to the data frame
  all_results <- rbind(all_results, c(var, beta, hr, ci_lower, ci_upper, p_value))
}

# Convert the results into a data frame and set column names
all_results <- as.data.frame(all_results)
colnames(all_results) <- c("Variable", "Beta", "HR", "CI_lower", "CI_upper", "p_value")

# Write results to an Excel file
write.xlsx(all_results, "1234univariate_cox_results_with_CI.xlsx")

# Print results
print(all_results)
